<?php

/**
 * @copyright
 * @package    Easy Joomla Backup Pro - EJB for Joomla! 5
 * @author     Viktor Vogel <admin@kubik-rubik.de>
 * @version    5.1.0.0-PRO - 2024-07-28
 * @link       https://kubik-rubik.de/ejb-easy-joomla-backup
 *
 * @license    GNU/GPL
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <https://www.gnu.org/licenses/>.
 */
defined('_JEXEC') || die('Restricted access');

use Joomla\CMS\{Factory, Version, Component\ComponentHelper, Installer\Installer, Language\Text, Table\Table, Uri\Uri};
use Joomla\Filesystem\File;
use Joomla\Registry\Registry;

/**
 * Class Com_EasyJoomlaBackupInstallerScript
 *
 * This class must be compatible with the minimum PHP requirement of Joomla! 5 -> PHP 8.1!
 * Newer PHP syntax cannot be used to avoid errors even before the PHP check has been executed.
 *
 * @version 5.1.0.0-PRO
 * @since   5.0.0.0-PRO
 */
class Com_EasyJoomlaBackupInstallerScript
{
    public const MAX_VERSION_JOOMLA = '6.0.0';
    public const MIN_VERSION_JOOMLA = '5.0.0';
    public const MIN_VERSION_PHP = '8.1.0';

    /**
     * Name of extension that is used in the error message
     *
     * @var string
     * @since 5.0.0.0-PRO
     */
    protected $extensionName = 'Easy Joomla Backup Pro';

    /**
     * Name of extension that is used in the error message
     *
     * @var string
     * @since 5.0.0.0-PRO
     */
    protected $extensionElement = 'com_easyjoomlabackup';

    /**
     * Checks compatibility in the preflight event
     *
     * @param string $type
     * @param object $parent
     *
     * @return bool
     * @throws Exception
     * @version 5.1.0.0-PRO
     * @since   5.0.0.0-PRO
     */
    public function preflight(string $type, object $parent): bool
    {
        if ($type === 'uninstall') {
            return true;
        }

        if (!$this->checkVersionJoomla()) {
            return false;
        }

        if (!$this->checkVersionPhp()) {
            return false;
        }

        return true;
    }

    /**
     * Checks whether the Joomla! version meets the requirement
     *
     * @return bool
     * @throws Exception
     * @since 5.0.0.0-PRO
     */
    private function checkVersionJoomla(): bool
    {
        $version = new Version();

        if (!$version->isCompatible(self::MIN_VERSION_JOOMLA)) {
            Factory::getApplication()->enqueueMessage(Text::sprintf('KRJE_ERROR_JOOMLA_VERSION', $this->extensionName, self::MIN_VERSION_JOOMLA), 'error');

            return false;
        }

        if (version_compare(JVERSION, self::MAX_VERSION_JOOMLA, 'ge')) {
            Factory::getApplication()->enqueueMessage(Text::sprintf('KRJE_ERROR_JOOMLA_VERSION_MAX', $this->extensionName, self::MAX_VERSION_JOOMLA), 'error');

            return false;
        }

        return true;
    }

    /**
     * Checks whether the PHP version meets the requirement
     *
     * @return bool
     * @throws Exception
     * @since 5.0.0.0-PRO
     */
    private function checkVersionPhp(): bool
    {
        if (!version_compare(PHP_VERSION, self::MIN_VERSION_PHP, 'ge')) {
            Factory::getApplication()->enqueueMessage(Text::sprintf('KRJE_ERROR_PHP_VERSION', $this->extensionName, self::MIN_VERSION_PHP), 'error');

            return false;
        }

        return true;
    }

    /**
     * @param object $parent
     *
     * @since 5.0.0.0-PRO
     */
    public function update(object $parent): void
    {
        $db = Factory::getContainer()->get('DatabaseDriver');
        $db->setQuery('DESCRIBE ' . $db->quoteName('#__easyjoomlabackup'));
        $dbStructure = $db->loadAssocList();

        $db->setQuery('ALTER TABLE ' . $db->quoteName('#__easyjoomlabackup') . ' CONVERT TO CHARACTER SET utf8mb4 COLLATE utf8mb4_unicode_ci');

        $db->setQuery('ALTER TABLE ' . $db->quoteName('#__easyjoomlabackup') . ' CHANGE ' . $db->quoteName('comment') . ' ' . $db->quoteName('comment') . ' TINYTEXT CHARACTER SET utf8mb4 COLLATE utf8mb4_unicode_ci NOT NULL');
        $db->execute();

        $db->setQuery('ALTER TABLE ' . $db->quoteName('#__easyjoomlabackup') . ' CHANGE ' . $db->quoteName('type') . ' ' . $db->quoteName('type') . ' VARCHAR(32) CHARACTER SET utf8mb4 COLLATE utf8mb4_unicode_ci NOT NULL');
        $db->execute();

        $db->setQuery('ALTER TABLE ' . $db->quoteName('#__easyjoomlabackup') . ' CHANGE ' . $db->quoteName('size') . ' ' . $db->quoteName('size') . ' VARCHAR(12) CHARACTER SET utf8mb4 COLLATE utf8mb4_unicode_ci NOT NULL');
        $db->execute();

        $db->setQuery('ALTER TABLE ' . $db->quoteName('#__easyjoomlabackup') . ' CHANGE ' . $db->quoteName('duration') . ' ' . $db->quoteName('duration') . ' VARCHAR(8) CHARACTER SET utf8mb4 COLLATE utf8mb4_unicode_ci NOT NULL');
        $db->execute();

        $db->setQuery('ALTER TABLE ' . $db->quoteName('#__easyjoomlabackup') . ' CHANGE ' . $db->quoteName('name') . ' ' . $db->quoteName('name') . ' TINYTEXT CHARACTER SET utf8mb4 COLLATE utf8mb4_unicode_ci NOT NULL');
        $db->execute();

        $manifest = $parent->getManifest();
        $parent = $parent->getParent();
        $source = $parent->getPath('source');

        $installer = new Installer();

        foreach ($manifest->plugins->plugin as $plugin) {
            $attributes = $plugin->attributes();
            $plg = $source . '/' . $attributes['folder'] . '/' . $attributes['plugin'];
            $installer->install($plg);
        }
    }

    /**
     * @param object $parent
     *
     * @since 5.0.0.0-PRO
     */
    public function install(object $parent): void
    {
        $manifest = $parent->getManifest();
        $parent = $parent->getParent();
        $source = $parent->getPath('source');

        $installer = new Installer();

        foreach ($manifest->plugins->plugin as $plugin) {
            $attributes = $plugin->attributes();
            $plg = $source . '/' . $attributes['folder'] . '/' . $attributes['plugin'];
            $installer->install($plg);
        }
    }

    /**
     * @param string $type
     * @param object $parent
     *
     * @since 5.0.0.0-PRO
     */
    public function postflight(string $type, object $parent): void
    {
        if ($type === 'uninstall') {
            return;
        }

        $proSecurityToken = $parent->getParent()->getPath('source') . '/ProSecurityToken.php';

        if (is_file($proSecurityToken)) {
            $destination = $parent->getParent()->getPath('extension_root') . '/ProSecurityToken.php';
            File::copy($proSecurityToken, $destination);
        }

        $db = Factory::getContainer()->get('DatabaseDriver');

        $db->setQuery('UPDATE ' . $db->quoteName('#__extensions') . ' SET ' . $db->quoteName('enabled') . ' = 1 WHERE ' . $db->quoteName('element') . " = 'easyjoomlabackupcronjob' AND " . $db->quoteName('type') . " = 'plugin'");
        $db->execute();

        $db->setQuery('UPDATE ' . $db->quoteName('#__extensions') . ' SET ' . $db->quoteName('enabled') . ' = 1 WHERE ' . $db->quoteName('element') . " = 'easyjoomlabackupcli' AND " . $db->quoteName('type') . " = 'plugin'");
        $db->execute();

        $componentParams = $this->getComponentParams();
        $componentId = ComponentHelper::getComponent('com_easyjoomlabackup')->id;

        if (!empty($componentParams) && !empty($componentId)) {
            $table = Table::getInstance('extension');
            $table->load($componentId);
            $table->bind(['params' => $componentParams->toString()]);

            if (!$table->check()) {
                return;
            }

            $table->store();
        }
    }

    /**
     * Gets the plugin parameters and sets a new security token
     *
     * @return bool|Registry
     * @since 5.0.0.0-PRO
     */
    private function getComponentParams()
    {
        $componentParams = ComponentHelper::getParams($this->extensionElement);

        if ($componentParams !== null) {
            $this->setUrlRoot($componentParams);

            return $componentParams;
        }

        return false;
    }

    /**
     * Sets the root URL required for the CLI script
     *
     * @param object $params
     *
     * @since 5.0.0.0-PRO
     */
    private function setUrlRoot(object $params): void
    {
        $params->set('urlRoot', Uri::root());
    }
}
