<?php
/**
 * @copyright	Copyright (C) 2011 Simplify Your Web, Inc. All rights reserved.
 * @license		GNU General Public License version 3 or later; see LICENSE.txt
 */

namespace SYW\Component\WeblinkLogosPro\Administrator\Table;

defined('_JEXEC') or die;

use Joomla\CMS\Factory;
use Joomla\CMS\Application\ApplicationHelper;
use Joomla\CMS\HTML\Helpers\StringHelper;
use Joomla\CMS\Language\Text;
use Joomla\CMS\String\PunycodeHelper;
use Joomla\CMS\Table\Table;
use Joomla\CMS\Tag\TaggableTableInterface;
use Joomla\CMS\Tag\TaggableTableTrait;
use Joomla\Database\DatabaseDriver;
//use Joomla\CMS\Versioning\VersionableTableInterface;
use Joomla\Utilities\ArrayHelper;
use Joomla\Registry\Registry;

/**
 * use weblinks Table class
 */
class WeblinkTable extends Table implements TaggableTableInterface
{
	use TaggableTableTrait;
	
	/**
	 * Indicates that columns fully support the NULL value in the database
	 *
	 * @var    boolean
	 */
	protected $_supportNullValue = true;

    /**
     * Constructor
     *
     * @param   JDatabase  &$db  A database connector object
     */
	public function __construct(DatabaseDriver $db)
    {
		$this->typeAlias = 'com_weblinks.weblink';

        parent::__construct('#__weblinks', 'id', $db);
    }

    /**
     * Overloaded bind function to pre-process the params.
     *
     * @param   array  $array   Named array
     * @param   mixed  $ignore  Optional array or list of parameters to ignore
     *
     * @return  null|string  null is operation was satisfactory, otherwise returns an error
     *
     * @see     Table:bind
     * @since   1.5
     */
    public function bind($array, $ignore = '')
    {
        $date = Factory::getDate();
        $task = Factory::getApplication()->input->get('task');

        if ($array['id'] == 0 && empty($array['created_by'])) {
            $array['created_by'] = Factory::getUser()->id;
        }

        if (isset($array['params']) && is_array($array['params'])) {
            $registry = new Registry();
            $registry->loadArray($array['params']);
            $array['params'] = (string) $registry;
        }

        if (isset($array['metadata']) && is_array($array['metadata'])) {
            $registry = new Registry();
            $registry->loadArray($array['metadata']);
            $array['metadata'] = (string) $registry;
        }

//         if (!Factory::getUser()->authorise('core.admin', 'com_weblinklogospro.weblink.' . $array['id']))
//         {
//             $actions = JAccess::getActionsFromFile(
//                 JPATH_ADMINISTRATOR . '/components/com_weblinklogospro/access.xml',
//                 "/access/section[@name='weblink']/"
//                 );
//             $default_actions = JAccess::getAssetRules('com_weblinklogospro.weblink.' . $array['id'])->getData();
//             $array_jaccess = array();

//             foreach ($actions as $action) {
//                 if (key_exists($action->name, $default_actions)) {
//                     $array_jaccess[$action->name] = $default_actions[$action->name];
//                 }
//             }

//             $array['rules'] = $this->JAccessRulestoArray($array_jaccess);
//         }

        // Bind the rules for ACL where supported.
        if (isset($array['rules']) && is_array($array['rules'])) {
            $this->setRules($array['rules']);
        }

        return parent::bind($array, $ignore);
    }

    /**
     * This function convert an array of JAccessRule objects into an rules array.
     *
     * @param   array  $jaccessrules  An array of JAccessRule objects.
     *
     * @return  array
     */
    private function JAccessRulestoArray($jaccessrules)
    {
        $rules = array();

        foreach ($jaccessrules as $action => $jaccess)
        {
            $actions = array();

            if ($jaccess)
            {
                foreach ($jaccess->getData() as $group => $allow)
                {
                    $actions[$group] = ((bool)$allow);
                }
            }

            $rules[$action] = $actions;
        }

        return $rules;
    }

    /**
     * Overloaded check function
     *
     * @return bool
     */
    public function check()
    {
        try {
            parent::check();
        } catch (\Exception $e) {
            $this->setError($e->getMessage());
            
            return false;
        }
        
        // If there is an ordering column and this is a new row then get the next ordering value
        if (property_exists($this, 'ordering') && $this->id == 0) {
            $this->ordering = self::getNextOrder();
        }

        // Check for valid name
        if (trim($this->title) == '') {
            $this->setError(Text::_('COM_WEBLINKLOGOSPRO_WARNING_PROVIDE_VALID_TITLE'));
            return false;
        }

        // Generate a valid alias
        $this->generateAlias();

        // Check for valid category
        if (trim($this->catid) == '') {
            $this->setError(Text::_('COM_WEBLINKLOGOSPRO_WARNING_CATEGORY'));
            return false;
        }
        
        // Check the publish down date is not earlier than publish up.
        if ((int) $this->publish_down > 0 && $this->publish_down < $this->publish_up)
        {
            $this->setError(Text::_('JGLOBAL_START_PUBLISH_AFTER_FINISH'));
            
            return false;
        }
        
        /*
         * Clean up keywords -- eliminate extra spaces between phrases
         * and cr (\r) and lf (\n) characters from string
         */
        if (!empty($this->metakey))
        {
            // Array of characters to remove
            $bad_characters = array("\n", "\r", "\"", "<", ">");
            $after_clean    = StringHelper::str_ireplace($bad_characters, "", $this->metakey);
            $keys           = explode(',', $after_clean);
            $clean_keys     = array();
            
            foreach ($keys as $key)
            {
                // Ignore blank keywords
                if (trim($key))
                {
                    $clean_keys[] = trim($key);
                }
            }
            
            // Put array back together delimited by ", "
            $this->metakey = implode(", ", $clean_keys);
        }
        
        /**
         * Ensure any new items have compulsory fields set. This is needed for things like
         * frontend editing where we don't show all the fields or using some kind of API
         */
        if (!$this->id)
        {
            if (!isset($this->xreference))
            {
                $this->xreference = '';
            }
            
            if (!isset($this->metakey))
            {
                $this->metakey = '';
            }
            
            if (!isset($this->metadesc))
            {
                $this->metadesc = '';
            }
            
            if (!isset($this->images))
            {
                $this->images = '{}';
            }
            
            if (!isset($this->metadata))
            {
                $this->metadata = '{}';
            }
            
            if (!isset($this->params))
            {
                $this->params = '{}';
            }
        }

        return true;
    }

    /**
     * Stores a weblink
     *
     * @param   boolean  $updateNulls  True to update fields even if they are null.
     *
     * @return  boolean  True on success, false on failure.
     *
     * @since   1.6
     */
    public function store($updateNulls = false)
    {
        // Transform the params field
        if (is_array($this->params)) {
            $registry = new Registry($this->params);
            $this->params = (string) $registry;
        }

        // Transform the images field
        if (is_array($this->images)) {
            $registry = new Registry($this->images);
            $this->images = (string) $registry;
        }

        $date = Factory::getDate()->toSql();

        $user = Factory::getUser();

        $this->modified = $date;

        if ($this->id) { // Existing item
            $this->modified_by = $user->id;
        } else { // New weblink
            if (!(int) $this->created) {
                $this->created = $date;
            }

            if (empty($this->created_by)) {
                $this->created_by = $user->id;
            }
        }

        // Set publish_up to null date if not set
        if (!$this->publish_up) {
            $this->publish_up = NULL;
        }

        // Set publish_down to null date if not set
        if (!$this->publish_down) {
            $this->publish_down = NULL;
        }

        // Set xreference to empty string if not set
        if (!$this->xreference) {
            $this->xreference = '';
        }

        // Convert IDN urls to punycode
        $this->url = PunycodeHelper::urlToPunycode($this->url);

        // Verify that the alias is unique
        $table = new WeblinkTable($this->getDbo());

        if ($table->load(array('alias' => $this->alias, 'catid' => $this->catid)) && ($table->id != $this->id || $this->id == 0)) {
            $this->setError(Text::_('COM_WEBLINKLOGOSPRO_ERROR_UNIQUE_ALIAS'));

            return false;
        }

        return parent::store($updateNulls);
    }

    /**
     * Method to set the publishing state for a row or list of rows in the database
     * table.  The method respects checked out rows by other users and will attempt
     * to checkin rows that it can after adjustments are made.
     *
     * @param   mixed    $pks     An optional array of primary key values to update.  If not
     *                            set the instance property value is used.
     * @param   integer  $state   The publishing state. eg. [0 = unpublished, 1 = published]
     * @param   integer  $userId  The user id of the user performing the operation.
     *
     * @return   boolean  True on success.
     *
     * @since    1.0.4
     *
     * @throws \Exception
     */
    public function publish($pks = null, $state = 1, $userId = 0)
    {
        // Initialise variables.
        $k = $this->_tbl_key;

        // Sanitize input.
        ArrayHelper::toInteger($pks);
        $userId = (int) $userId;
        $state  = (int) $state;

        // If there are no primary keys set check to see if the instance key is set.
        if (empty($pks))
        {
            if ($this->$k)
            {
                $pks = array($this->$k);
            }
            // Nothing to set publishing state on, return false.
            else
            {
                throw new \Exception(500, Text::_('JLIB_DATABASE_ERROR_NO_ROWS_SELECTED'));
            }
        }

        // Build the WHERE clause for the primary keys.
        $where = $k . '=' . implode(' OR ' . $k . '=', $pks);

        // Determine if there is checkin support for the table.
        if (!property_exists($this, 'checked_out') || !property_exists($this, 'checked_out_time'))
        {
            $checkin = '';
        }
        else
        {
            $checkin = ' AND (checked_out = 0 OR checked_out = ' . (int) $userId . ')';
        }

        // Update the publishing state for rows with the given primary keys.
        $this->_db->setQuery(
        	'UPDATE ' . $this->db->quoteName($this->_tbl) .
            ' SET `published` = ' . (int) $state .
            ' WHERE (' . $where . ')' .
            $checkin
            );
        $this->_db->execute();

        // If checkin is supported and all rows were adjusted, check them in.
        if ($checkin && (count($pks) == $this->_db->getAffectedRows()))
        {
            // Checkin each row.
            foreach ($pks as $pk)
            {
                $this->checkin($pk);
            }
        }

        // If the JTable instance value is in the list of primary keys that were set, set the instance.
        if (in_array($this->$k, $pks))
        {
            $this->state = $state;
        }

        return true;
    }

    /**
     * Define a namespaced asset name for inclusion in the #__assets table
     *
     * @return string The asset name
     *
     * @see Table::_getAssetName
     */
    protected function _getAssetName()
    {
        $k = $this->_tbl_key;

        return 'com_weblinklogospro.weblink.' . (int) $this->$k;
    }

    /**
     * Returns the parent asset's id. If you have a tree structure, retrieve the parent's id using the external key field
     *
     * @param   Table   $table  Table name
     * @param   integer  $id     Id
     *
     * @see Table::_getAssetParentId
     *
     * @return mixed The id on success, false on failure.
     */
    protected function _getAssetParentId(Table $table = null, $id = null)
    {
        // We will retrieve the parent-asset from the Asset-table
        $assetParent = Table::getInstance('Asset');

        // Default: if no asset-parent can be found we take the global asset
        $assetParentId = $assetParent->getRootId();

        // The item has the component as asset-parent
        $assetParent->loadByName('com_weblinklogospro');

        // Return the found asset-parent-id
        if ($assetParent->id)
        {
            $assetParentId = $assetParent->id;
        }

        return $assetParentId;
    }

    /**
     * Delete a record by id
     *
     * @param   mixed  $pk  Primary key value to delete. Optional
     *
     * @return bool
     */
    public function delete($pk = null)
    {
        $this->load($pk);
        $result = parent::delete($pk);

        return $result;
    }

    /**
     * Generate a valid alias
     * Remains public to be able to check for duplicated alias before saving
     *
     * @return  string
     */
    public function generateAlias()
    {
        if (empty($this->alias))
        {
            $this->alias = $this->title;
        }

        $this->alias = ApplicationHelper::stringURLSafe($this->alias, $this->language);

        if (trim(str_replace('-', '', $this->alias)) == '')
        {
            $this->alias = Factory::getDate()->format('Y-m-d-H-i-s');
        }

        return $this->alias;
    }

    /**
     * Get the type alias for the history table
     *
     * @return  string  The alias as described above
     */
    public function getTypeAlias()
    {
    	return 'com_weblinks.weblink';
    }
}
